/*****************************************************************************
* AN_Adding flexibility  to MKW01 by implementing Whitening and CRC by SW
*                            *Application Demo*
* 
* Copyright (c) 2014, Freescale, Inc. All rights reserved.
*
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
*
*                      *Simple Range Demo main file*
* The code for this Application Note is based on the simple range demo application. 
* The simple range demonstration runs as a standalone application that allows 
* performing dynamic range tests.
*
* The simple demonstration consists of two nodes:
*
* TX Node
* RX Node
*
* In addition to the simple range demonstration functionality a Packet Error Rate 
* (PER) test was implemented. The PER test allows the user via serial terminal 
* interface to evaluate the performance of the communication between two devices. 
* The demo can be run without a serial terminal as a range test demo.  If the user 
* wants to evaluate PER test then the serial terminal interface is needed. 
*
*
*****************************************************************************/

#include "ApplicationConf.h"                                                    /*Defines the Application default parameters*/
#include "SMAC.h"
#include "FunctionLib.h"
#include "Keyboard_interface.h"
#include "crc_ibm.h"                                                            // To import CRC calculation function
#include "DataWhitening.h"                                                      // To import Data Whitening function

/************************************************************************************
*************************************************************************************
* Private prototypes 
*************************************************************************************
************************************************************************************/

void MCPSDataConfirm(txStatus_t TransmissionResult);
void MCPSDataIndication(rxPacket_t *gsRxPacket);

void TimerDelayCallback(tmrTimerID_t timerId);
void Keyboard_CallBack(uint8_t Keyboard_PortID);

void PlatformPortInit(void);
void InitGlobalVariables(void);
void Platform_Init(void);
void SetDefaultValues(void);
void InitSmac(void);

void IncrementSeqNumber(void);                                                  // Increment sequence number function
void ResetSeqNumber(void);                                                      // Reset sequence number function
void CopyCRCToAppPayloadTx (void);                                              // Copy CRC bytes to Tx array function

void PacketErrorRateTx(void);                                                   // Packet error rate function for TX
void PacketErrorRateRx(void);                                                   // Packet error rate function for RX


/************************************************************************************
*************************************************************************************
* Public memory declarations                                                                   
*************************************************************************************
************************************************************************************/
#define gRxNode_c 			     (2)
#define gTxNode_c                            (3)
 
/*Defines used in App Demo*/

#define AppPayload_SeqNumberByte1ArrayPosition          3                       // Position of sequence number byte 1 in data frame
#define AppPayload_SeqNumberByte2ArrayPosition          4                       // Position of sequence number byte 2 in data frame
#define AppPayload_SeqNumberByte3ArrayPosition          5                       // Position of sequence number byte 3 in data frame
#define AppPayload_SeqNumberByte4ArrayPosition          6                       // Position of sequence number byte 4 in data frame

#define gSimpleRangeDefaultPayloadLength_c   (20)                               // 1 byte for Payload lenght + 17 Payload Bytes + 2 Bytes for CRC 

tmrTimerID_t delayTimer = gTmrInvalidTimerID_c;

static uint8_t gau8RxDataBuffer[gMaxSmacSDULength_c]; 
static uint8_t gau8TxDataBuffer[gMaxSmacSDULength_c]; 
 
static txPacket_t * gAppTxPacket;
static rxPacket_t * gAppRxPacket;

bool_t bTxDone;
bool_t bRxDone;
bool_t bDelayTimeout;
bool_t bTimeout;

#define PerTotalPackets_c  20                                                   // Total Packets for Packet error rate


/*Data variables for demo*/
uint8_t AppPayload_Tx [gSimpleRangePayloadLength_c] = {0x11, 0x5A, 0xA5,0x00,0x00,0x00,0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x00, 0x00}; //Payload Array for Demo (Last 2 bytes reserved for CRC)
uint8_t AppRxBuffer [gSimpleRangePayloadLength_c];                              // Buffer to store payload data
uint8_t index;                                                                  // Index to iterate in the bucle where the data of reception is stored.
uint16_t CRC_Calculation_Tx;                                                    // Variable where CRC calculation is stored to then be inserted in AppPayloadTx

typedef enum perTxStates_tag 
{
  gPerTxStateIdle_c,
  gPerTxStateStartTest_c,
  gPerTxStateRunningTest_c
}perTxStates_t;                                                                 // Tx PER States definition

typedef enum perRxStates_tag 
{
  gPerRxStateIdle_c,
  gPerRxStateReceiving_c
}perRxStates_t;                                                                 // Rx PER States definition

perTxStates_t perTxState;                                                       // Tx Per state variable declaration
perRxStates_t perRxState;                                                       // Rx Per state variable declaration

uint16_t u16SentPackets;                                                        // Variable to count packets transmitted in PER

bool_t bStartTxPER;                                                             // Start Per TX Flag
bool_t bStartRxPER;                                                             // Start Per RX Flag

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/

/************************************************************************************
*
* main
*
************************************************************************************/

void main(void)
{
  
  DisableInterrupts();
  /* Initialization of MCU and transceiver  */
  Platform_Init();                                                              
  InitGlobalVariables();                                                        
  /* SMAC Initialization */
  InitSmac();
  /* Set Radio to default values */
  SetDefaultValues();      
  EnableInterrupts();                                                           
 
  #if (gRxNode_c == gDeviceType_c)
               
             TurnOnLeds();                                                      // Turn on and off leds ONCE to indicate is a RX device type 
             Delay_ms(500);                                                     // '' ''
             TurnOffLeds();                                                     // '' ''             
  
             UartUtil_Print("\r\n\nPress switch 1 to start PER (Rx)\r\n", gAllowToBlock_d);  // Print instructions to user by uart 
             
             perRxState = gPerRxStateIdle_c;                                    // PER RX state machine init
           
  #endif
           
  #if (gTxNode_c == gDeviceType_c) 
           
           TurnOnLeds();                                                        // Turn on and off leds TWICE to indicate is a TX device type 
           Delay_ms(500);                                                       // '' ''
           TurnOffLeds();                                                       // '' ''        
           Delay_ms(500);                                                       // '' ''
           TurnOnLeds();                                                        // '' ''
           Delay_ms(500);                                                       // '' ''
           TurnOffLeds();                                                       // '' ''
                             
           UartUtil_Print("\r\n\nPress switch 1 to start PER (Tx)\r\n", gAllowToBlock_d);  // Print instructions to user by uart 
           
           perTxState = gPerTxStateIdle_c;                                      // PER TX state machine init 
         
  #endif 
  
  for(;;)
  {
    #if (gRxNode_c == gDeviceType_c)
                                                                               
         PacketErrorRateRx();                                                   // Packet Error Rate state machine for RX 
                                 
    #endif
                     
    #if (gTxNode_c == gDeviceType_c)

        if (bStartTxPER)                                                        // Check if user pressed the button to start the test
        {     
            PacketErrorRateTx();                                                // Packet Error Rate state machine for TX   
        }      
    
    #endif    
	   
  } 
}
          


void PacketErrorRateTx(void)
{
 
    switch(perTxState)
    {
       case gPerTxStateIdle_c:
                 
           UartUtil_Print("Running PER (Tx). Sending ", gAllowToBlock_d);       // Print status of the application to user by uart 
           PrintWordOnDecimalFormatBlocking(PerTotalPackets_c, 0, FALSE, 0);    // '' ''
           UartUtil_Print(" packets\r\n", gAllowToBlock_d);                     // '' ''
           TurnOffLeds();                                                       //
           Led4On();                                                            // Turn on led to indicate that the application started
           perTxState = gPerTxStateStartTest_c;                                 // 
        
       break; 
       case gPerTxStateStartTest_c:
        
           CRC_Calculation_Tx = RadioComputeCRC(AppPayload_Tx, 18, CRC_TYPE_IBM);            // Calculate CRC from the first 18 payload bytes
           CopyCRCToAppPayloadTx();                                                          // Copy CRC calculation to App Payload frame 
           FLib_MemCpy(&gAppTxPacket->u8DataLength, AppPayload_Tx, sizeof(AppPayload_Tx));   // Fill gAppTxPacket with AppPayload_Tx data
           //RadioComputeWhitening(&gAppTxPacket->u8DataLength, 20);                           // Data whitening IBM applied to all 20 bytes of the gAppTxPacket payload frame
           RadioComputeWhitening_CCIT(&gAppTxPacket->u8DataLength, 20);                           // Data whitening CCIT applied to all 20 bytes of the gAppTxPacket payload frame
           (void)MCPSDataRequest(gAppTxPacket);                                              // Transmit the packet   
           TMR_StartTimer(delayTimer, gTmrSingleShotTimer_c, TmrMilliseconds(gDelayBetweenPacketsInMs_c), TimerDelayCallback); // Run a timer to have a delay between packets 
           IncrementSeqNumber();                                                // Increment sequence number for each packet transmitted
           u16SentPackets++;                                                    // Increment counting of packets sent
           Led1Toggle();                                                        // Toggle led to indicate a packet was sent
           perTxState = gPerTxStateRunningTest_c;                               // Go to gPerTxStateRunningTest_c state
      
       break;  
       case gPerTxStateRunningTest_c:
           
           if(bDelayTimeout)                                                    // Check if time delay flag has been activated
           {
                if(bTxDone)                                                     // Check if TxDone flag has been activated
                {   
                    bTxDone = FALSE;                                            // Clean TxDne flag
                   
                    if (u16SentPackets == PerTotalPackets_c)                    // Check if all packets has been transmitted
                    {
                        UartUtil_Print("PER Finished (Tx).\r\n\nPress switch 1 to start PER (Tx)\r\n", gAllowToBlock_d); // Print instructions to user by uart 
                        u16SentPackets = 0;                                     // Clean variable to count packets sent
                        TurnOffLeds();                                          // Turn off leds to indicate the application finished
                        ResetSeqNumber();                                       // Reset the sequence number to 00's
                        bStartTxPER = FALSE;                                    // Clean start flag
                        perTxState = gPerTxStateIdle_c;                         // Go to gPerTxStateIdle_c state                
                    }
                    else
                    {
                        CRC_Calculation_Tx = RadioComputeCRC(AppPayload_Tx, 18, CRC_TYPE_IBM);            // Calculate CRC from the first 18 payload bytes (last 2 bytes are reserved to insert CRC after be calculated)
                        CopyCRCToAppPayloadTx();                                                          // Copy the CRC calculation to AppPayload_Tx array   
                        FLib_MemCpy(&gAppTxPacket->u8DataLength, AppPayload_Tx, sizeof(AppPayload_Tx));   // Fill gAppTxPacket Buffer (it is the buffer that is going to be transmitted)
                        //RadioComputeWhitening(&gAppTxPacket->u8DataLength, 20);                           // Data whitening IBM applied to all 20 bytes of the gAppTxPacket payload frame
                        RadioComputeWhitening_CCIT(&gAppTxPacket->u8DataLength, 20);                           // Data whitening CCIT applied to all 20 bytes of the gAppTxPacket payload frame
                        (void)MCPSDataRequest(gAppTxPacket);                                              // Transmit the packet      
                        TMR_StartTimer(delayTimer, gTmrSingleShotTimer_c, TmrMilliseconds(gDelayBetweenPacketsInMs_c), TimerDelayCallback); // Run a timer to have a delay between packets
                        IncrementSeqNumber();                                   // Increment sequence number
                        u16SentPackets++;                                       // Increment Packets sent variable
                        Led1Toggle();                                           // Toggle a led to indicate a packet has been transmitted
                    }
                }
             bDelayTimeout = FALSE;                                             // Clean Delay flag
           }          
           
       break; 
       default:
       break;
    }
  
}  

void PacketErrorRateRx(void)
{
    switch(perRxState) 
    {
    case gPerRxStateIdle_c:
      
         while (!bStartRxPER);                                                  // Wait for user press button to start application
         bStartRxPER = FALSE;                                                   // Clean start flag
         UartUtil_Print("Running PER (Rx). Receiving packets\r\n", gAllowToBlock_d);   // Print status of the application to user by uart  
         u16ReceivedPackets = 0;                                                // Received packets variable init
         (void)MLMERXEnableRequest(gAppRxPacket, 0);                            // Enable RX
         Led4On();                                                              // Indicate the used the application started
         perRxState = gPerRxStateReceiving_c;                                   // Go to gPerRxStateReceiving_c state
         
    break;
    case gPerRxStateReceiving_c:
             
         if (bRxDone)                                                           // Check if a message has been received
         {
            bRxDone = FALSE;                                                    // Clean RxDone flag
            (void)MLMERXEnableRequest(gAppRxPacket, 0);                         // Enable RX
         }
         
         if (u16PacketsIndex == PerTotalPackets_c)                              // Check if all packets has been received
         {    
             UartUtil_Print("PER finished (Rx)\r\n", gAllowToBlock_d);          // Print status to user by uart 
             PrintWordOnDecimalFormatBlocking(u16ReceivedPackets, 0, FALSE, 0); // Print results to user by uart
             UartUtil_Print(" of ", gAllowToBlock_d);                           // '' ''       
             PrintWordOnDecimalFormatBlocking(PerTotalPackets_c, 0, FALSE, 0);  // '' ''
             UartUtil_Print(" packets received successfully\r\n\nPress switch 1 to start PER (Rx)\r\n", gAllowToBlock_d);  // '' '' 
             u16PacketsIndex = 0;                                               // Clean packets index variable
             perRxState = gPerRxStateIdle_c;                                    // Go to gPerRxStateIdle_c state
             TurnOffLeds();                                                     // Turn off leds to indicate the user the application has finished
             (void)MLMERXDisableRequest();                                      // Disable RX
         }
         
    break;
    default:
    break;
    }
}


 void SetDefaultValues(void)
   {
     
         (void) PhyPib_SetCarrierFreq (gDefaultRfFreq_c);                       //915 MHz for US. - 866.525 Mhz for EU
	 PhyPib_SetBitrate(gDefaultBitRate_c);                                  //50000 bps
	 PhyPib_SetFreqDeviation(gDefaultFreqDv_c);                             //25000 Hz                        
 	 (void)PhyPib_SetModulationType(gDefaultModulationType_c);              //FSK
 	 (void)PhyPib_SetModulationShaping(gDefaultModulationShaping_c);        //BT_1
	 (void)PhyPib_EnableSequencer(gDefaultSequencer_c);                     //ON
	 (void)PhyPib_SetDataMode(gDefaultDataMode_c);                          //Packet_Mode
         PhyPib_SetPowerStep(gDefaultOutputPower);                              //PA0 output power -1dBm for US. Commented for EU (13dBm default). Enabled for US
	 PhyPib_EnableAfcAutoclear(gDefaultAfcAutoClear_c);                     //FALSE
	 PhyPib_EnableAfcAuto(gDefaultAfcAuto_c);                               //FALSE
	 PhyPib_SetRxTimeout(gDefaultRxTimeout_c);                              //0x00
	 PhyPib_SetRxTimeoutThreshold(gDefaultRxTimeoutThresh_c);               //0x00
	 (void)PhyPib_EnableSensitivityBoost(gDefaultSensitivityBoost_c);       //Normal Sensitivity
	 PhyPib_SetPaRamp(gDefaultPaRamp_c);            
	 PhyPib_SetLnaGain(gDefaultLnaGain_c);
	 PhyPib_SetRssiThreshold(gDefaultRssiThresh_c);                         //0xFF
	 PhyPib_SetDccFreq(gDefaultDccFreq_c);                                  //DccFreq_7
	 PhyPib_SetRxFilterBw(gDefaultRxFilterBw_c);                            //83,300
	 PhyPib_SetDccFreqAfc(gDefaultDccFreqAfc_c);                            //DccFreq_5                          
         PhyPib_SetRxFilterBwAfc(gDefaultRxFilterBwAfc_c);                      //100,000
         /* New Sync Word values for demo*/
         PhyPib_SetSyncWordValue(MKW01_Reg_SyncValue1, 0xF4);
         PhyPib_SetSyncWordValue(MKW01_Reg_SyncValue2, 0xEE);
         PhyPib_SetSyncWordValue(MKW01_Reg_SyncValue3, 0xF4);
         PhyPib_SetSyncWordValue(MKW01_Reg_SyncValue4, 0xEE);
	 PhyPib_SetFifoThreshold(gDefaultFifoThresh_c);
         
         
        /*Settings for Wide Band Systems*/
        /*Refer to chapter Low Modulation Index Systems in the device datasheet for more details*/
         MKW01Drv_WriteRegister(MKW01_Reg_TestDagc, 0x30);                      //Improved margin, used when AfcLowBetaOn=0
         MKW01Drv_WriteRegister(MKW01_Reg_AfcCtrl, 0x00);                       //Standard AFC routine, for signals with modulation index (2*Fdev/BR) greater than 2.
         MKW01Drv_WriteRegister(MKW01_Reg_RssiThresh, 0xE4);                    //Default(-114 dBms).
         MKW01Drv_WriteRegister(MKW01_Reg_AfcFei, 0x18);                        //AfcAutoClearOn=1. AfcAutoOn=0 (AFC only performed after AfcStart is set. (No AFC for each packet, only once to compensate for the Tx frequency deviation)                   
   }

void Platform_Init(void) 
 {

 volatile uint8_t dummy;
 
        /* Init MCU specifics: SPI - and reset Radio */
        PlatformPortInit();                                                     
        LED_Init();                                                                                                   
       	MKW01Drv_SpiInit();                                                     // Init SPI for Radio driver
        
	/* Use external clock for testing */
	dummy = MKW01Drv_ReadRegister(0x59);
	dummy &= 0xEF;
	dummy |= 0x10;
	MKW01Drv_WriteRegister(0x59, dummy);   
         
 	(void)MKW01Drv_Reset();					                // Reset Radio  
        
        MKW01Drv_ConfigureCLKO(ClkOutFxOsc_Div16);                              //32Mhz/16= 2MHz CLKOUT
                   
        /*Initialize clock based on CLKOUT*/
        MCG_Setup();                                         
        MKW01Drv_WriteRegister(MKW01_Reg_PaRamp, 0x0A);           

        /* Timer module initialization */
        TMR_Init();  
        TMR_Task(); 
        
        /*Keyboard module initialization*/
        Keyboard_Init(Keyboard_CallBack);
        
        /* Uart module initialization */
        Uart_ModuleInit();
 }


/************************************************************************************
*
* InitGlobalVariables
*
************************************************************************************/

void InitGlobalVariables(void)
{

  /* Initialize Global Variables */ 
   uint8_t payloadFill = 0;
   uint8_t payloadIndex = 27;

   bRxDone = FALSE;
   bTxDone = FALSE;
   bTimeout = FALSE;
   bDelayTimeout = FALSE;	 
    
  delayTimer = TMR_AllocateTimer();
   
   /* Initialization of SMAC variables */
   gAppRxPacket = (rxPacket_t *)gau8RxDataBuffer;
   gAppRxPacket->u8MaxDataLength = gMaxSmacSDULength_c;  
   
   gAppTxPacket = (txPacket_t *)gau8TxDataBuffer;   
   payloadFill = gSimpleRangePayloadLength_c - gSimpleRangeDefaultPayloadLength_c;
   while(payloadFill--)                                                         // If payload set by the user bigger than default 
   {
           gAppTxPacket->smacPdu.smacPdu[payloadIndex++] = payloadFill;
   }      
}

/************************************************************************************
*
* InitSmac
*
************************************************************************************/

void InitSmac(void)
{
            
   smacStandalone = FALSE;
   MLMERadioInit();  
   MLMESetPreambleLength(gDefaultPreambleLength_c);
   MLMESetSyncWordSize(gDefaultSyncWordSize_c);
   
#ifdef MKW01_NA
   MLMESetFreqBand(g902__928MHz_c, gRFMode0_c);                                 //Americas configuration (915 MHz)
#endif
#ifdef MKW01_EU
   MLMESetFreqBand(g863__870MHz_c, gRFMode0_c);                                 //Europe configuration (868.3MHz MHz). Changed in SetDefaultValues() to 868.525 MHz
#endif

}

 void PlatformPortInit(void)
 {
    /* Enable all of the port clocks. These have to be enabled to configure
     * pin muxing options, so most code will need all of these on anyway.
     */
    SIM_SCGC5 |= (SIM_SCGC5_PORTA_MASK
                  | SIM_SCGC5_PORTB_MASK
                  | SIM_SCGC5_PORTC_MASK
                  | SIM_SCGC5_PORTD_MASK
                  | SIM_SCGC5_PORTE_MASK );

    /*Need to set pin as input*/
    /* Port A, Pin 18 <- EXTAL */ 
    /* pins direction -> INPUT (PDD&=~0x20000) */
    PORTA_PCR18 = PORT_PCR_MUX(1);
    GPIOA_PDDR &= (uint32_t)~(uint32_t)(gMKW01ClkOutMask_c);

    /*Need to set IRQ pins as input*/ 
    /*For Radio pins DIO0, DIO1, DIO4*/
    MKW01xDrv_IRQ_PortConfig();
    GPIOC_PDDR &= (uint32_t)~(uint32_t)(1<<3);      //PTC3<-DIO0                
    GPIOC_PDDR &= (uint32_t)~(uint32_t)(1<<4);      //PTC4<-DIO1                
    GPIOD_PDDR &= (uint32_t)~(uint32_t)(1<<4);      //PTD4<-DIO4         
        
 }


/************************************************************************************
* User's Callbacks
************************************************************************************/

/* Place your callbacks here */

/************************************************************************************
*
* TimerDelayCallback                                                            
*
************************************************************************************/
  void TimerDelayCallback(tmrTimerID_t timerId)
  {
	  bDelayTimeout = TRUE;
  }

/************************************************************************************
*
* Keyboard_CallBack                                                                    
*
************************************************************************************/
  void Keyboard_CallBack(uint8_t Keyboard_PortID)
  {

    
    #if (gTxNode_c == gDeviceType_c)  
    
      if(Keyboard_PortID == 1)                                             
       {
            bStartTxPER = TRUE;                                                 // Flag to start PER TX 
       }
      
    #endif
        
    #if (gRxNode_c == gDeviceType_c)    
      if(Keyboard_PortID == 1)                                             
       {  
            bStartRxPER = TRUE;                                                 // Flag to start PER RX 
       }
    #endif
      
  }

/************************************************************************************
* SMAC Callbacks
************************************************************************************/


/************************************************************************************
*
* MCPSDataConfirm
*
************************************************************************************/

void MCPSDataConfirm(txStatus_t TransmissionResult)
{
  bTxDone = TRUE;
}

/************************************************************************************
*
* MCPSDataIndication
*
************************************************************************************/

void MCPSDataIndication(rxPacket_t *gsRxPacket)
{  
   if(rxSuccessStatus_c == gsRxPacket->rxStatus)
   {
    bRxDone = TRUE;	    
   }
   else if (rxTimeOutStatus_c == gsRxPacket->rxStatus)
   {
    bTimeout = TRUE;
   }
}

/************************************************************************************
*
* ResetSeqNumber 
*
************************************************************************************/

void ResetSeqNumber(void){

     AppPayload_Tx[AppPayload_SeqNumberByte1ArrayPosition  ] = 0;               // Reset to 0s payload frame sequence number 
     AppPayload_Tx[AppPayload_SeqNumberByte2ArrayPosition  ] = 0;
     AppPayload_Tx[AppPayload_SeqNumberByte3ArrayPosition  ] = 0;
     AppPayload_Tx[AppPayload_SeqNumberByte4ArrayPosition  ] = 0;

}


/************************************************************************************
*
* IncrementSeqNumber 
*
************************************************************************************/

void IncrementSeqNumber(void){
   
	if (AppPayload_Tx [AppPayload_SeqNumberByte1ArrayPosition  ] == 0xFF){
            AppPayload_Tx[AppPayload_SeqNumberByte1ArrayPosition  ] = 0;
            if (AppPayload_Tx [AppPayload_SeqNumberByte2ArrayPosition  ] == 0xFF){
                AppPayload_Tx[AppPayload_SeqNumberByte2ArrayPosition  ] = 0;
                if (AppPayload_Tx [AppPayload_SeqNumberByte3ArrayPosition  ] == 0xFF){
                    AppPayload_Tx[AppPayload_SeqNumberByte3ArrayPosition  ] = 0;
                    if (AppPayload_Tx [AppPayload_SeqNumberByte4ArrayPosition  ] == 0xFF){
                       AppPayload_Tx[AppPayload_SeqNumberByte4ArrayPosition  ] = 0;  
                    }else{
                         AppPayload_Tx [AppPayload_SeqNumberByte4ArrayPosition  ]++;
                    }  
                }else{
                     AppPayload_Tx [AppPayload_SeqNumberByte3ArrayPosition  ] ++;
                }
            }else{
                  AppPayload_Tx [AppPayload_SeqNumberByte2ArrayPosition  ] ++;
            }    

	}else{
             AppPayload_Tx [AppPayload_SeqNumberByte1ArrayPosition  ] ++;
        }    

}

void CopyCRCToAppPayloadTx(void){

  AppPayload_Tx [18] = (uint8_t) (CRC_Calculation_Tx>>8);     
  AppPayload_Tx [19] = (uint8_t) CRC_Calculation_Tx;        //  Fill Tx Payload Buffer with CRC bytes.     
  
}



